// contentScript.js

function debugPrint(message) {
  console.log("ContentScript:", message);
}

function clickButton(buttonId) {
  let button = document.getElementById(buttonId);
  if (button) {
    button.click();
    debugPrint(`${buttonId} button clicked.`);
  } else {
    debugPrint(`Button with ID ${buttonId} not found.`);
  }
}

function fillTextField(fieldId, data) {
  let textField = document.getElementById(fieldId);
  if (textField) {
    textField.value = data;
    debugPrint(`Text field with ID ${fieldId} filled with data: ${data}`);
  } else {
    debugPrint(`Text field with ID ${fieldId} not found.`);
  }
}

async function waitForElement(selector, timeout = 5000) {
  const startTime = new Date().getTime();
  return new Promise((resolve, reject) => {
    const checkInterval = setInterval(() => {
      const currentTime = new Date().getTime();
      const element = document.querySelector(selector);
      if (element) {
        clearInterval(checkInterval);
        resolve(element);
      } else if (currentTime - startTime > timeout) {
        clearInterval(checkInterval);
        document.location = "https://office.sud.kz/form/proceedings/myCases.xhtml";
        chrome.runtime.sendMessage({ action: 'startOver' });
        reject(`Timeout waiting for element: ${selector}`);
      }
    }, 300);
  });
}

async function clickListItem() {
  try {
    debugPrint("Checking for empty list message...");
    const emptyListMessage = document.querySelector("#j_idt113\\:requestPanel").textContent.trim();
    if (emptyListMessage.includes("Сіздің тізіміңіз бос") || emptyListMessage.includes("Ваш список пуст")) {
      setResult([{ dateTime: "", content: "Дело с таким номером не существует" }]);
      return;
    }
    debugPrint("Waiting for the list item...");
    const listItem = await waitForElement("#j_idt113\\:requestPanel > ul > li");
    listItem.scrollIntoView();
    saveExtensionState('extractingData');
    listItem.click();
    debugPrint("List item clicked.");
  } catch (e) {
    console.error("Error trying to click the list item: ", e);
  }
}

async function extractData() {
  try {
    debugPrint("Extracting data from the new page...");
    await waitForElement(".panel-body");
    const rows = document.querySelectorAll("div.panel-body > div.well");
    const data_list = Array.from(rows).flatMap(row => {
      const dateTimeFirstFormat = row.querySelector("div > div > p:nth-of-type(2)");
      const mainContentFirstFormat = row.querySelector("div[style='margin-left: 150px;']");
      if (dateTimeFirstFormat && mainContentFirstFormat) {
        return [{
          dateTime: dateTimeFirstFormat.textContent,
          mainContent: mainContentFirstFormat.textContent
        }];
      }
      const dateTime = row.querySelector(".label.label-info").textContent.trim();
      const mainContent = Array.from(row.childNodes)
        .filter(node => node.nodeType === Node.TEXT_NODE)
        .map(node => node.textContent.trim()).join(" ").trim();
      return [{ dateTime, mainContent }];
    });

    let res = [];
    data_list.forEach(data => {
      res.push({ dateTime: data.dateTime, content: data.mainContent });
    });

    const backLink = await waitForElement("a[href='/form/proceedings/myCases.xhtml']");
    backLink.click();
    debugPrint("Navigated back.");
    await setResult(res);
    chrome.runtime.sendMessage({ action: 'responseReady', res: res });
    debugPrint("Navigating back to the list...");
    saveExtensionState('start');
  } catch (error) {
    console.error("An error occurred:", error);
  }
}

function waitForLoader() {
  return new Promise((resolve, reject) => {
    let loaderAppearTimeout;
    let loaderDisappearInterval;
    const waitForLoaderToAppear = () => {
      const loaderIsActive = document.querySelector("div.blockUI.blockMsg.blockPage");
      if (loaderIsActive) {
        clearInterval(loaderAppearInterval);
        clearTimeout(loaderAppearTimeout);
        waitForLoaderToDisappear();
      }
    };
    const waitForLoaderToDisappear = () => {
      loaderDisappearInterval = setInterval(() => {
        const loaderIsActive = document.querySelector("div.blockUI.blockMsg.blockPage");
        if (!loaderIsActive) {
          clearInterval(loaderDisappearInterval);
          resolve();
        }
      }, 100);
    };
    const loaderAppearInterval = setInterval(waitForLoaderToAppear, 300);
    loaderAppearTimeout = setTimeout(() => {
      clearInterval(loaderAppearInterval);
      clearInterval(loaderDisappearInterval);
      chrome.runtime.sendMessage({ action: 'startOver' });
      reject("Loader timeout");
    }, 1500);
  });
}

async function clickButtonAndWaitForLoader(buttonId) {
  clickButton(buttonId);
  await waitForLoader();
}

function getExtensionState() {
  return new Promise((resolve) => {
    chrome.runtime.sendMessage({ action: 'getState' }, function(response) {
      resolve(response.state);
    });
  });
}

function saveExtensionState(state) {
  return new Promise((resolve) => {
    chrome.runtime.sendMessage({ action: 'setState', state }, function(response) {
      resolve(response.state);
    });
  });
}

function getDataSet(addData) {
  return new Promise((resolve) => {
    chrome.runtime.sendMessage({ action: 'getDataS'}, function(response) {
      resolve(response.data);
    });
  });
}

function setResult(res) {
  return new Promise((resolve) => {
    chrome.runtime.sendMessage({ action: 'setRes', res }, function(response) {
      resolve();
    });
  });
}

async function main() {
  const state = await getExtensionState();
  const data = await getDataSet();
  if (state === 'start' || state === 'startOver') {
    if (data === null) return;
    await processData(data);
  } else if (state === 'extractingData') {
    await extractData();
  }
}

async function processData(data) {
  console.log(data);
  await clickButtonAndWaitForLoader("j_idt113:filterBtn");
  console.log('clicked')
  fillTextField("j_idt345:filter-requestNumber", data);
  console.log("ss")
  await clickButtonAndWaitForLoader("j_idt345:j_idt360");
  saveExtensionState('extractingData');
  await clickListItem();
}

// === Kazpatent Easywork Flag Logic ===
if (window.location.hostname === 'newcab.kazpatent.kz') {
  chrome.storage.local.get(['kazpatent_from_easywork'], (result) => {
    console.log('[Kazpatent] storage.local.get:', result);
    if (result.kazpatent_from_easywork) {
      showKazpatentUI();
      chrome.storage.local.remove('kazpatent_from_easywork');
    }
  });
}

// === Kazpatent Clicker Variables ===
let kazpatentClickerActive = false;
let kazpatentClickCount = 0;
let kazpatentTotalRuns = 0;
let kazpatentXpaths = [];
let kazpatentCurrentIndex = 0;

// === Kazpatent Clicker Functions ===
async function loadKazpatentXpaths() {
  try {
    const response = await fetch(chrome.runtime.getURL('xpath.txt'));
    const text = await response.text();
    kazpatentXpaths = text.split('\n')
      .map(line => line.trim())
      .filter(line => line.length > 0)
      .map(line => line.replace(/^['"]|['"]$/g, '')); // Убираем кавычки в начале и конце
    console.log(`[Kazpatent] Загружено ${kazpatentXpaths.length} XPath селекторов`);
    return kazpatentXpaths.length;
  } catch (error) {
    console.error('[Kazpatent] Ошибка загрузки XPath файла:', error);
    return 0;
  }
}

function incrementKazpatentTotalRuns() {
  kazpatentTotalRuns++;
  return kazpatentTotalRuns;
}

function updateKazpatentUI() {
  const panel = document.getElementById('kazpatentClickerPanel');
  if (panel) {
    const statusDiv = panel.querySelector('.status');
    if (statusDiv) {
      statusDiv.innerHTML = `
        <div style="margin-top:8px; font-size:12px; color:#666;">
          Прогресс: ${kazpatentClickCount}/${kazpatentXpaths.length}<br>
          Запусков: ${kazpatentTotalRuns}
        </div>
      `;
    }
  }
}

async function kazpatentClickWithDelay(index) {
  if (!kazpatentClickerActive || index >= kazpatentXpaths.length) {
    console.log('[Kazpatent] Кликер остановлен или завершен');
    kazpatentClickerActive = false;
    updateKazpatentUI();
    return;
  }

  try {
    const xpath = kazpatentXpaths[index];
    console.log(`[Kazpatent] Клик ${index + 1}/${kazpatentXpaths.length}: ${xpath}`);
    
    // Находим элемент по XPath
    const element = document.evaluate(
      xpath,
      document,
      null,
      XPathResult.FIRST_ORDERED_NODE_TYPE,
      null
    ).singleNodeValue;

    if (element) {
      // Прокручиваем к элементу
      element.scrollIntoView({ behavior: 'smooth', block: 'center' });
      
      // Небольшая задержка для прокрутки
      await new Promise(resolve => setTimeout(resolve, 500));
      
      // Кликаем по элементу
      element.click();
      kazpatentClickCount++;
      updateKazpatentUI();
      
      console.log(`[Kazpatent] Успешный клик по элементу ${index + 1}`);
    } else {
      console.log(`[Kazpatent] Элемент не найден: ${xpath}`);
    }

    // Задержка перед следующим кликом (400мс)
    const delay = 400;
    console.log(`[Kazpatent] Следующий клик через ${delay}мс`);
    
    setTimeout(() => {
      kazpatentClickWithDelay(index + 1);
    }, delay);

  } catch (error) {
    console.error(`[Kazpatent] Ошибка при клике ${index + 1}:`, error);
    // Продолжаем со следующим элементом
    setTimeout(() => {
      kazpatentClickWithDelay(index + 1);
    }, 1000);
  }
}

function stopKazpatentClicker() {
  console.log('[Kazpatent] Остановка кликера');
  kazpatentClickerActive = false;
  updateKazpatentUI();
}

async function startKazpatentClicker() {
  console.log('[Kazpatent] Запуск кликера');
  
  // Загружаем XPath селекторы
  const xpathCount = await loadKazpatentXpaths();
  if (xpathCount === 0) {
    alert('Не удалось загрузить XPath селекторы!');
    return;
  }

  // Сбрасываем счетчики
  kazpatentClickCount = 0;
  kazpatentCurrentIndex = 0;
  kazpatentClickerActive = true;
  
  // Увеличиваем счетчик запусков
  const totalRuns = incrementKazpatentTotalRuns();
  
  // Обновляем UI
  updateKazpatentUI();
  
  console.log(`[Kazpatent] Начинаем кликать по ${xpathCount} элементам`);
  
  // Запускаем кликер
  kazpatentClickWithDelay(0);
}

function showKazpatentUI() {
  console.log('[Kazpatent] showKazpatentUI called');
  
  // Удаляем старую панель, если есть
  const oldPanel = document.getElementById('kazpatentClickerPanel');
  if (oldPanel) oldPanel.remove();
  
  // Создаем новую панель управления кликером
  const panel = document.createElement('div');
  panel.id = 'kazpatentClickerPanel';
  panel.style.position = 'fixed';
  panel.style.top = '40px';
  panel.style.right = '20px';
  panel.style.zIndex = 10000;
  panel.style.background = '#fffbe6';
  panel.style.border = '1px solid #ffc107';
  panel.style.borderRadius = '8px';
  panel.style.padding = '16px 20px 12px 20px';
  panel.style.boxShadow = '0 2px 8px rgba(0,0,0,0.15)';
  panel.style.fontFamily = 'inherit';
  panel.style.minWidth = '200px';
  
  panel.innerHTML = `
    <div style="font-weight:bold; margin-bottom:8px; color:#b8860b;">Кликер Kazpatent</div>
    <div style="margin-bottom:12px;">
      <button id="startClickerBtn" style="margin-right:8px; background:#52c41a; color:#fff; border:none; border-radius:4px; padding:6px 16px; cursor:pointer; font-size:15px;">Старт</button>
      <button id="stopClickerBtn" style="background:#ff4d4f; color:#fff; border:none; border-radius:4px; padding:6px 16px; cursor:pointer; font-size:15px;">Стоп</button>
    </div>
    <div class="status" style="margin-top:8px; font-size:12px; color:#666;">
      Готов к запуску
    </div>
  `;
  
  document.body.appendChild(panel);

  // Добавляем обработчики событий
  document.getElementById('startClickerBtn').onclick = () => {
    startKazpatentClicker();
  };
  
  document.getElementById('stopClickerBtn').onclick = () => {
    stopKazpatentClicker();
  };
}

main();
